/**
    \file SimpleList.cpp
    Implementation of SimpleList distance transform (simply an exhaustive 
    search employing a list) class which, given and input binary image, 
    calculates the corresponding distance transform.

    \author George J. Grevera, Ph.D., ggrevera@sju.edu

    Copyright (C) 2002, George J. Grevera

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
    USA or from http://www.gnu.org/licenses/gpl.txt.

    This General Public License does not permit incorporating this
    code into proprietary programs.  (So a hypothetical company such
    as GH (Generally Hectic) should NOT incorporate this code into
    their proprietary programs.)
 */
/*
     no OUT OF BOUNDS checks!
*/
#include <stdlib.h>
#include <vector>
#include "SimpleList.h"

using namespace std;
//----------------------------------------------------------------------
/*
Input : (C,I) - a 2D binary scene of domain size X by Y
Output: (C,d) - a 2D grey scene of domain size X and Y representing the
                distance scene
*/
void SimpleList::doTransform ( const unsigned char* const I ) {
    borderCheck(I);
    cleanUp();
    int x, y;

    //initialize d
    double* d = (double*)malloc( ySize*xSize*sizeof(double)   );
    assert( d!=NULL );
    this->p = (P*)malloc( ySize*xSize*sizeof(P) );
    assert( this->p!=NULL );
    for (y=0; y<ySize; y++) {
        for (x=0; x<xSize; x++) {
            const int i=sub(x,y);
            d[i] = DistanceTransform::FloatInfinity;
            //pointer to immediate interior or exterior i.e.
            //p(x,y).x is the x coordinate of the closest boundary
            //point (and similarly for p(x,y).y)
            this->p[i].x = this->p[i].y = -1;
        }
    }

    //initialize immediate interior & exterior elements (i.e. border elements)
    vector<P*>  v;
    //double halfDx = 0, halfDy = 0;
    for (y=1; y<ySize-1; y++) {
        for (x=1; x<xSize-1; x++) {
            if ( I[sub(x-1,y)] != I[sub(x,y)] || I[sub(x+1,y)] != I[sub(x,y)] ||
                 I[sub(x,y-1)] != I[sub(x,y)] || I[sub(x,y+1)] != I[sub(x,y)] ) {
                    const int i=sub(x,y);
                    d[i] = 0;
                    this->p[i].x = x;
                    this->p[i].y = y;
                    P* p = new P(x,y);
                    v.push_back(p);
            }
        }
    }

    //iterate over all (non border element) points
    for (y=1; y<ySize-1; y++) {
        for (x=1; x<xSize-1; x++) {
            const int c = sub(x,y);
            if (d[c]==0)    continue;  //skip border elements
            //at this stage, we have a point that is not an element of 
            // the border.  iterate over all border elements.
            for (int i=0; i<(int)v.size(); i++) {
                const int x1 = v[i]->x;
                const int y1 = v[i]->y;
                //calculate the distance to this border element
                const double dTmp = sqrt( (double)
                    ((x-x1)*(x-x1) + (y-y1)*(y-y1)) );
                //is it better than what's already been assigned?
                if (dTmp < d[c]) {
                    //yes, then change to this border element
                    d[c] = dTmp;
                    this->p[c].x = x1;
                    this->p[c].y = y1;
                }
            }
        }  //end for x
    }  //end for y

    finish(I, d);
}
//----------------------------------------------------------------------
